import ReactFlow, {
  Controls,
  Background,
  useNodesState,
  useEdgesState,
  Handle,
  Position,
  ReactFlowProps,
  Node,
} from "react-flow-renderer";
import { useCallback, useEffect, useMemo, useState } from "react";
import classNames from "classnames";

import jobFail from "@/assets/project/jobFail.svg";
import jobRun from "@/assets/project/jobRun.svg";
import jobSue from "@/assets/project/jobSue.svg";
import { IEdge, IExecutionStatus, ITask } from "../../ProjectSubmitWork/interface";
import { IBatchNode, ILine } from "./interface";

import styles from "./index.module.css";
/*
 * @Author: 吴永生#A02208 yongsheng.wu@wholion.com
 * @Date: 2022-06-22 10:15:22
 * @LastEditors: 吴永生#A02208 yongsheng.wu@wholion.com
 * @LastEditTime: 2022-06-22 11:10:35
 * @FilePath: /bkunyun/src/views/Project/components/Flow/index.tsx
 * @Description: 这是默认设置,请设置`customMade`, 打开koroFileHeader查看配置 进行设置: https://github.com/OBKoro1/koro1FileHeader/wiki/%E9%85%8D%E7%BD%AE
 */
interface IProps extends ReactFlowProps {
  tasks?: ITask[];
  /** 点击batch事件 */
  onBatchClick?: (val: string) => void;
}

  /** 获取imgUrl */
  const getImgUrl = (type: IExecutionStatus) => {
    if(type === 'Done'){
      return jobSue
    }
    if(type === 'Failed'){
      return jobFail
    }
    if(type === 'Running'){
      return jobRun
    }
    return undefined
  }

/** 自定义batch节点 */
const BatchNode = (props: IBatchNode) => {
  const { data } = props;
  const { dotStatus, style, isFlowNode, label, selectedStatus } = data;
  return (
    <div
      className={classNames({
        [styles.batchNode]: true,
        [styles.selectedBatchBox]: selectedStatus,
        [styles.selectBatchNode]: selectedStatus,
      })}
      style={style}
    >
      {dotStatus?.isInput ? (
        <Handle
          style={{ background: "#fff ", border: "1px solid #D1D6DE", left: 20 }}
          type="target"
          position={Position.Top}
        />
      ) : null}
      <div
        className={classNames({
          [styles.batchRotate]: isFlowNode,
        })}
      >
        {label || ""}
        {data.isCheck && <span className={styles.successDot}></span>}
      </div>
      {dotStatus?.isOutput ? (
        <Handle
          style={{ background: "#fff ", border: "1px solid #D1D6DE", left: 20 }}
          type="source"
          position={Position.Bottom}
        />
      ) : null}
    </div>
  );
};

/** 自定义flow节点 */
const FlowNode = (props: any) => {
  const { data } = props;
  return (
    <div
      className={classNames({
        [styles.flowNode]: true,
      })}
    >
      {data?.dotStatus?.isInput ? (
        <Handle
          style={{ background: "#C2C6CC ", left: 12 }}
          type="target"
          position={Position.Top}
        />
      ) : null}
      <div style={{ display: "flex", alignItems: "center" }}>
        {data?.label || ""}
        {data.isCheck && <span className={styles.successDot}></span>}
        {getImgUrl(data.executionStatus) && <img style={{ marginLeft: "6px" }} src={getImgUrl(data.executionStatus)} alt="" />}
      </div>
      {data?.dotStatus?.isOutput ? (
        <Handle
          style={{ background: "#C2C6CC ", left: 12 }}
          type="source"
          position={Position.Bottom}
          id="a"
        />
      ) : null}
    </div>
  );
};

const Flow = (props: IProps) => {
  const { tasks, onBatchClick } = props;
  console.log(tasks,'tasks')
  const [selectedNodeId, setSelectedNodeId] = useState<string>("");
  /** 自定义的节点类型 */
  const nodeTypes = useMemo(() => {
    return { batchNode: BatchNode, flowNode: FlowNode };
  }, []);

  /** 获取是否有输入节点或者是否有输出节点 */
  const nodesInputAndOutputStatus = useCallback(
    (id: string) => {
      /** 所有的连线 */
      const lineArr: IEdge[] = [];
      tasks?.length &&
        tasks.forEach((item) => {
          lineArr.push(...item.edges);
        });
      /** 所有的输入节点ID */
      const isInput = lineArr?.some((item) => item.target === id);
      /** 所有的输出节点ID */
      const isOutput = lineArr?.some((item) => item.source === id);
      return {
        isInput,
        isOutput,
      };
    },
    [tasks]
  );

  /** 获取是否有流节点 */
  const isFlowNode = useCallback(
    (id: string) => {
      return (
        tasks?.length &&
        tasks?.some((item) => {
          return item.parentNode === id;
        })
      );
    },
    [tasks]
  );

  /** 通过子flow节点计算batch节点的样式 */
  const getBatchStyle = useCallback(
    (value: ITask) => {
      const positionXArr: number[] = [];
      const positionYArr: number[] = [];
      tasks?.length &&
        tasks?.forEach((item) => {
          if (item.parentNode === value.id) {
            positionXArr.push(item.position?.x || 0);
            positionYArr.push(item.position?.y || 0);
          }
        });
      positionXArr.sort((a, b) => {
        return a - b;
      });
      positionYArr.sort((a, b) => {
        return a - b;
      });
      let width = 176,
        height = 22;
      if (positionXArr?.length) {
        const val = positionXArr[positionXArr.length - 1] + 150;
        width = val > 176 ? val : width;
      }
      if (positionYArr?.length) {
        const val = positionYArr[positionYArr.length - 1];
        height = val > 22 ? val : height;
      }
      return {
        width,
        height,
      };
    },
    [tasks]
  );

  /** 生成初始化node节点 */
  const initialNodes = useMemo(() => {
    const val: any = [];
    tasks?.length &&
      tasks.forEach((item) => {
        val.push({
          id: item.id,
          type: item.type === "BATCH" ? "batchNode" : "flowNode",
          data: {
            label: item.title || "",

            ...(item.type === "BATCH"
              ? {
                  /** 是否有流节点 */
                  isFlowNode: isFlowNode(item.id),
                  /** 选中状态 */
                  selectedStatus: selectedNodeId === item.id,
                }
              : {}),
            isCheck: item.isCheck,
            executionStatus: item.executionStatus,
            /** 输入输出圆点状态 */
            dotStatus: nodesInputAndOutputStatus(item.id),
            /** 样式 */
            style: {
              ...getBatchStyle(item),
              padding: isFlowNode(item.id) ? "20px" : "12px 20px",
            },
          },
          position: { x: Number(item.position.x), y: Number(item.position.y) },
          ...(item.type === "BATCH" ? { style: { zIndex: -1 } } : {}),
          ...(item.parentNode ? { parentNode: item.parentNode } : {}),
          ...(item.type === "BATCH" ? { extent: "parent" } : {}),
        });
      });
    return val;
  }, [
    tasks,
    isFlowNode,
    selectedNodeId,
    nodesInputAndOutputStatus,
    getBatchStyle,
  ]);

  /** 生成初始化的连线节点 */
  const initialEdges = useMemo(() => {
    const val: ILine[] = [];
    tasks?.length &&
      tasks.forEach((item) => {
        item.edges.forEach((every) => {
          const newLine = {
            ...every,
            batchId: item.parentNode ? item.parentNode : item.id,
          };
          val.push(newLine);
        }, []);
      });
    return val.map((item: ILine) => {
      return {
        id: item.id,
        source: item.source,
        target: item.target,
        type: "smoothstep",
        ...(item?.batchId === selectedNodeId
          ? { style: { stroke: "#1370FF" }, animated: true }
          : {}),
        labelStyle: { fill: "#8A9099" },
        labelBgStyle: { fill: "#F7F8FA " },
        label: item.label ? `(${item.label})` : "",
      };
    });
  }, [selectedNodeId, tasks]);

  /** flowNode点击事件 */
  const onNodeClick = (e: any, node: Node) => {
    tasks?.forEach((item) => {
      if (item.id === node.id) {   
        if (item.parentNode) {
          setSelectedNodeId(item.parentNode);
          onBatchClick && onBatchClick(item.parentNode);
        } else {
          setSelectedNodeId(node.id);
          onBatchClick && onBatchClick(node.id || "");
        }
      }
    });
  };

  const handlePaneClick = () => {
    setSelectedNodeId('');
    onBatchClick && onBatchClick('');
  }

  /** node节点 */
  const [nodes, setNodes] = useNodesState(initialNodes);
  /** 连线数组 */
  const [edges, setEdges] = useEdgesState(initialEdges);

  useEffect(() => {
    setEdges(initialEdges);
  }, [initialEdges, setEdges]);

  useEffect(() => {
    setNodes(initialNodes);
  }, [initialNodes, setNodes]);

  return (
    <ReactFlow
      nodes={nodes}
      edges={edges}
      fitView
      proOptions={{ hideAttribution: true, account: "" }}
      nodeTypes={nodeTypes}
      onPaneClick={handlePaneClick}
      onNodeClick={onNodeClick}
      {...props}
    >
      <Controls />
      <Background  color="#aaa" gap={16} />
    </ReactFlow>
  );
};

export default Flow;
